@extends('user.layout.sub-app')
@section('title', 'Brochure | Extrutech Hardware')

{{-- 
    NOTE: $sitesetting variable is assumed to be available and contain 
    company details like comp_name, comp_address, comp_phone, etc. 
    You might need to pass $sitesetting from your controller.
--}}
@php
    // --- Dynamic SEO Variables for Brochure Page ---
    $seo = [
        'title' => 'Download Extrutech Hardware Brochure & Catalogue',
        'description' => 'Explore our latest collection of profiles, hardware, and architectural solutions. Download the official Extrutech Hardware brochure now.',
        'keywords' => 'extrutech brochure, hardware catalogue, aluminum profiles PDF, architectural hardware download',
        'image' => asset('assets/user/img/extrutech.png'), // Use an appropriate brochure cover image
        'author' => 'Extrutech Hardware',
        'robots' => 'INDEX,FOLLOW',
        'canonical' => url('/brochure') // Assuming the URL is /brochure
    ];
@endphp
@section('meta')
<meta property="og:type" content="website">
<meta property="og:title" content="{{ $seo['title'] }}">
<meta property="og:description" content="{{ $seo['description'] }}">
<meta property="og:image" content="{{ $seo['image'] }}">
<meta property="og:url" content="{{ $seo['canonical'] }}">
<meta property="og:site_name" content="{{ $sitesetting->comp_name ?? 'Extrutech Hardware' }}">

<meta name="twitter:card" content="summary_large_image">
<meta name="twitter:title" content="{{ $seo['title'] }}">
<meta name="twitter:description" content="{{ $seo['description'] }}">
<meta name="twitter:image" content="{{ $seo['image'] }}">
@endsection


@section('schema')
<script type="application/ld+json">
{
  "@context": "https://schema.org",
  "@type": "WebPage",
  "name": "{{ $seo['title'] }}",
  "description": "{{ $seo['description'] }}",
  "url": "{{ $seo['canonical'] }}",
  "potentialAction": {
    "@type": "DownloadAction",
    "name": "Download Brochure",
    "target": {
      "@type": "EntryPoint",
      "urlTemplate": "{{ url('/download-brochure') }}" // Replace with the actual download link
    }
  }
}
</script>
@endsection

@section('content')
    <div class="breadcumb-area" itemscope itemtype="https://schema.org/BreadcrumbList">
        <div class="breadcumb-wrapper py-3" data-bg-src="{{ asset('assets/user/img/bg/breadcumb-bg.jpg') }}">
            <div class="container">
                <div class="breadcumb-content">
                    <h1 class="breadcumb-title text-anime" style="font-size: 34px;" itemprop="name">Download Brochure</h1>
                    <ul class="breadcumb-menu">
                        <li class="wow fadeInUp" data-wow-delay=".2s" itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem">
                            <a href="{{ url('/') }}" itemprop="item"><span itemprop="name">Home</span></a>
                            <meta itemprop="position" content="1" />
                        </li>
                        <li class="wow fadeInUp" data-wow-delay=".3s" itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem">
                            <span itemprop="name">Brochure</span>
                            <meta itemprop="position" content="2" />
                        </li>
                    </ul>
                </div>
            </div>
        </div>
    </div>

    <div class="brochure_section d-md-block d-none position-relative py-3">
        <div id="loader"><i class="fas fa-spinner fa-spin"></i> Loading PDF...</div>
        <div id="canvas-container"></div>

        <div class="toolbar">
            <button class="tool-btn" onclick="zoomOut()"><i class="fas fa-minus"></i></button>
            <button class="tool-btn" onclick="zoomIn()"><i class="fas fa-plus"></i></button>
            <div class="separator"></div>
            <button class="tool-btn" id="btn-prev" onclick="prev()"><i class="fas fa-chevron-left"></i></button>
            <div class="page-info">
            <input id="page-input" type="text" value="1" onchange="jumpToPage(this.value)">
            <span>/ <span id="total-pages">--</span></span>
            </div>
            <button class="tool-btn" id="btn-next" onclick="next()"><i class="fas fa-chevron-right"></i></button>
            <div class="separator"></div>
            <button class="tool-btn" onclick="toggleFullScreen()"><i class="fas fa-expand"></i></button>
        </div>
    </div>
@endsection
@section('js')
<script type="module">
    import * as THREE from "three";

    const PDF_PATH = "{{ asset($brochure->pdf ?? 'assets/user/img/test1234.pdf') }}";
    const PAGE_WIDTH = 4.2;
    const PAGE_HEIGHT = 6.0;
    const FLIP_DURATION = 1200; // ms

    let scene, camera, renderer;
    let bookGroup;
    let pages = [];
    let totalPDFPages = 0;
    let currentSheetIndex = 0; // Tracks which physical sheet is currently active

    init();

    function init() {
      const container = document.getElementById("canvas-container");
      scene = new THREE.Scene();
    //   scene.background = new THREE.Color(0xD3D3D3);

      camera = new THREE.PerspectiveCamera(40, window.innerWidth / window.innerHeight, 0.1, 100);
      camera.position.set(0, 0, 10);

      renderer = new THREE.WebGLRenderer({ antialias: true, alpha: true });
      renderer.setPixelRatio(window.devicePixelRatio || 1);
      renderer.setSize(window.innerWidth, window.innerHeight);
      container.appendChild(renderer.domElement);

      loadPDFAndBuild();
      animate();
      window.addEventListener("resize", onWindowResize);
    }


    function flipCanvasHorizontally(srcCanvas) {
      const w = srcCanvas.width, h = srcCanvas.height;
      const out = document.createElement('canvas');
      out.width = w; out.height = h;
      const octx = out.getContext('2d');
      octx.save();
      octx.translate(w, 0);
      octx.scale(-1, 1);
      octx.drawImage(srcCanvas, 0, 0);
      octx.restore();
      return out;
    }

    async function renderPdfPageToCanvas(doc, pageNum) {
      const page = await doc.getPage(pageNum);
      const viewport = page.getViewport({ scale: 2.5 });
      const canvas = document.createElement('canvas');
      canvas.width = viewport.width; canvas.height = viewport.height;
      const ctx = canvas.getContext('2d');
      ctx.fillStyle = '#ffffff'; ctx.fillRect(0, 0, canvas.width, canvas.height);
      await page.render({ canvasContext: ctx, viewport }).promise;
      ctx.strokeStyle = '#cccccc';
      ctx.lineWidth = Math.max(2, Math.round(Math.min(canvas.width, canvas.height) * 0.003));
      ctx.strokeRect(0, 0, canvas.width, canvas.height);
      return canvas;
    }

    async function loadPDFAndBuild() {
      try {
        const doc = await pdfjsLib.getDocument(PDF_PATH).promise;
        totalPDFPages = doc.numPages;
        document.getElementById("total-pages").innerText = totalPDFPages;

        bookGroup = new THREE.Group();
        bookGroup.position.x = 0; 
        scene.add(bookGroup);

        const totalSheets = Math.ceil(totalPDFPages / 2);

        for (let i = 0; i < totalSheets; i++) {
            const frontNum = (i * 2) + 1;
            const backNum = (i * 2) + 2;
            let frontCanvas = await renderPdfPageToCanvas(doc, frontNum);
            const frontTex = new THREE.CanvasTexture(frontCanvas);
            frontTex.colorSpace = THREE.SRGBColorSpace;
            frontTex.minFilter = THREE.LinearFilter;

            let backTex;
            if (backNum <= totalPDFPages) {
                let backCanvas = await renderPdfPageToCanvas(doc, backNum);
                backCanvas = flipCanvasHorizontally(backCanvas);
                backTex = new THREE.CanvasTexture(backCanvas);
                backTex.colorSpace = THREE.SRGBColorSpace;
                backTex.minFilter = THREE.LinearFilter;
            } else {
                const white = new Uint8Array([255, 255, 255]);
                backTex = new THREE.DataTexture(white, 1, 1, THREE.RGBFormat);
                backTex.needsUpdate = true;
            }


            const sheet = createSheetMesh(frontTex, backTex, i);
            sheet.position.z = -i * 0.005; 
            
            pages.push(sheet);
            bookGroup.add(sheet);
        }

        document.getElementById("loader").style.display = "none";
        updateUI();

      } catch (e) {
        console.error(e);
        document.getElementById("loader").innerText = "Error: " + e.message;
      }
    }

    function createSheetMesh(frontTex, backTex, index) {
        const pivotGroup = new THREE.Group();

        const geo = new THREE.PlaneGeometry(PAGE_WIDTH, PAGE_HEIGHT, 15, 1);
        geo.translate(PAGE_WIDTH / 2, 0, 0); 
        
        const pos = geo.attributes.position;
        geo.userData = { originalPos: pos.array.slice() };

        const matFront = new THREE.MeshBasicMaterial({ map: frontTex, side: THREE.FrontSide });
        const meshFront = new THREE.Mesh(geo, matFront);
        meshFront.position.z = 0.001; 

        const matBack = new THREE.MeshBasicMaterial({ map: backTex, side: THREE.BackSide });
        const meshBack = new THREE.Mesh(geo, matBack);
        meshBack.position.z = 0.001;

        pivotGroup.add(meshFront);
        pivotGroup.add(meshBack);

        pivotGroup.userData = {
            index: index,
            targetAngle: 0, // 0 = Right (Open), -Math.PI = Left (Closed/Flipped)
            currentAngle: 0,
            isFlipping: false,
            flipStartTime: 0,
            startAngle: 0
        };

        return pivotGroup;
    }

    function animate() {
      requestAnimationFrame(animate);

      const now = performance.now();

      pages.forEach((sheet, i) => {
          const u = sheet.userData;
          
          if (u.isFlipping) {
              const elapsed = now - u.flipStartTime;
              let p = elapsed / FLIP_DURATION;
              if (p >= 1) {
                  p = 1;
                  u.isFlipping = false;
              }

              const q = p < 0.5 ? 4 * p * p * p : 1 - Math.pow(-2 * p + 2, 3) / 2;

              sheet.rotation.y = u.startAngle + (u.targetAngle - u.startAngle) * q;
              sheet.position.y = 0; 
              
              let startZ, endZ;
              if (u.startAngle > -1) { // Was Right
                  startZ = -i * 0.005;
                  endZ = (i - pages.length) * 0.005;
              } else { // Was Left
                  startZ = (i - pages.length) * 0.005;
                  endZ = -i * 0.005;
              }
              
              let currentZ = startZ + (endZ - startZ) * q;
              
              currentZ += Math.sin(q * Math.PI) * 0.01;
              
              sheet.position.z = currentZ;

              const curlStrength = Math.sin(q * Math.PI) * 1.5; 
              
              const mesh = sheet.children[0]; 
              const geo = mesh.geometry;
              const pos = geo.attributes.position;
              const original = geo.userData.originalPos;

              for(let k=0; k < pos.count; k++) {
                  const x = original[k*3];
                  const y = original[k*3+1];
                  const z = original[k*3+2];
                  const dist = x / PAGE_WIDTH;
                  const bendZ = (dist * dist) * curlStrength;
                  pos.setZ(k, z + bendZ);
              }
              pos.needsUpdate = true;

          } else {
              sheet.rotation.y = u.targetAngle;
              sheet.position.y = 0;

              const mesh = sheet.children[0];
              const geo = mesh.geometry;
              const pos = geo.attributes.position;
              const original = geo.userData.originalPos;
              
              for(let k=0; k < pos.count; k++) {
                  pos.setXYZ(k, original[k*3], original[k*3+1], original[k*3+2]);
              }
              pos.needsUpdate = true;
              if (u.targetAngle === 0) {
                  sheet.position.z = -i * 0.005; 
              } else {
                  sheet.position.z = (i - pages.length) * 0.005;
              }
          }
      });

      renderer.render(scene, camera);
    }


    window.next = function() {
        const sheet = pages.find(p => p.userData.targetAngle === 0);
        if (sheet) {
            triggerFlip(sheet, -Math.PI);
            updateUI();
        }
    };

    window.prev = function() {
        let sheetToFlip = null;
        for (let i = pages.length - 1; i >= 0; i--) {
            if (pages[i].userData.targetAngle <= -1) { 
                sheetToFlip = pages[i];
                break;
            }
        }

        if (sheetToFlip) {
            triggerFlip(sheetToFlip, 0);
            updateUI();
        }
    };

    window.jumpToPage = function(val) {
        const p = parseInt(val);
        if (!p || p < 1 || p > totalPDFPages) { updateUI(); return; }
        
        const targetSheetIndex = Math.floor((p - 1) / 2); 

        pages.forEach((sheet, i) => {
            let newTarget = 0;
            if (i < targetSheetIndex) {
                newTarget = -Math.PI;
            } else if (i === targetSheetIndex) {
                if (p % 2 === 0) newTarget = -Math.PI;
                else newTarget = 0;
            } else {
                newTarget = 0;
            }

            if (sheet.userData.targetAngle !== newTarget) {
                triggerFlip(sheet, newTarget);
            }
        });
        updateUI();
    };

    function triggerFlip(sheet, target) {
        const u = sheet.userData;
        if (u.targetAngle === target) return; // Already there or going there
        
        u.startAngle = sheet.rotation.y;
        u.targetAngle = target;
        u.flipStartTime = performance.now();
        u.isFlipping = true;
    }

    function updateUI() {
        if (!totalPDFPages) return;
        
        let lastFlippedIndex = -1;
        for(let i = 0; i < pages.length; i++) {
            if (pages[i].userData.targetAngle <= -1) lastFlippedIndex = i;
        }

        let displayVal;
        if (lastFlippedIndex === -1) {
            displayVal = "1";
        } else if (lastFlippedIndex === pages.length - 1) {
            displayVal = totalPDFPages.toString();
        } else {
            const leftP = (lastFlippedIndex * 2) + 2;
            const rightP = leftP + 1;
            displayVal = `${leftP}-${rightP}`;
        }

        const input = document.getElementById("page-input");
        const btnPrev = document.getElementById("btn-prev");
        const btnNext = document.getElementById("btn-next");
        
        input.value = displayVal;
        btnPrev.disabled = (lastFlippedIndex === -1);
        btnNext.disabled = (lastFlippedIndex === pages.length - 1);
    }

    window.zoomIn = function() { if (camera.position.z > 6) camera.position.z -= 2; };
    window.zoomOut = function() { if (camera.position.z < 60) camera.position.z += 2; };
    window.toggleFullScreen = function () {
      if (!document.fullscreenElement) document.documentElement.requestFullscreen();
      else if (document.exitFullscreen) document.exitFullscreen();
    };

    function onWindowResize() { camera.aspect = window.innerWidth / window.innerHeight; camera.updateProjectionMatrix(); renderer.setSize(window.innerWidth, window.innerHeight); }

  </script>
@endsection
